<?php

namespace App\Http\Controllers\Admin;

use App\Enums\Payment\GatewayType;
use App\Http\Controllers\Controller;
use App\Http\Requests\PaymentGatewayRequest;
use App\Services\Payment\PaymentGatewayService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Arr;
use Illuminate\View\View;

class ManualPaymentGatewayController extends Controller
{
    public function __construct(
        protected PaymentGatewayService $paymentGatewayService
    ) {}

    public function index(): View
    {
        $setTitle = __('admin.payment_processor.page_title.traditional.index');
        $gateways = $this->paymentGatewayService->getGatewayByPaginate(GatewayType::MANUAL);

        return view('admin.payment_gateway.manual.index', compact(
            'setTitle',
            'gateways'
        ));
    }

    public function create(): View
    {
        $setTitle = __('admin.payment_processor.page_title.traditional.create');
        return view('admin.payment_gateway.manual.create', compact('setTitle'));
    }

    public function store(PaymentGatewayRequest $request): RedirectResponse
    {
        $params = $this->paymentGatewayService->prepParams($request, GatewayType::MANUAL);
        
        // Handle QR code upload
        if ($request->hasFile('qr_code')) {
            $params['qr_code'] = $this->uploadQrCode($request->file('qr_code'));
        }
        
        $this->paymentGatewayService->save($params);
        
        return back()->with('notify', [
            ['success', __('admin.payment_processor.notify.traditional.create.success')]
        ]);
    }

    public function edit(string|int $id): View
    {
        $setTitle = __('admin.payment_processor.page_title.traditional.edit');
        $traditionalGateway = $this->paymentGatewayService->findById($id);
        
        abort_if(blank($traditionalGateway), 404);

        return view('admin.payment_gateway.manual.edit', compact(
            'setTitle',
            'traditionalGateway'
        ));
    }

    public function update(PaymentGatewayRequest $request, string|int $id): RedirectResponse
    {
        $traditionalGateway = $this->paymentGatewayService->findById($id);
        abort_if(blank($traditionalGateway), 404);

        $params = $this->paymentGatewayService->prepParams($request, GatewayType::MANUAL);
        
        // Handle QR code updates
        $this->handleQrCode($request, $traditionalGateway, $params);

        // Handle wallet address
        $params['wallet_address'] = $request->input('wallet_address');

        $traditionalGateway->update($params);

        return back()->with('notify', [
            ['success', __('admin.payment_processor.notify.traditional.update.success')]
        ]);
    }

    protected function handleQrCode($request, $gateway, &$params): void
    {
        // Remove existing QR code if requested
        if ($request->has('remove_qr')) {
            $this->deleteQrCode($gateway->qr_code);
            $params['qr_code'] = null;
        }

        // Upload new QR code if provided
        if ($request->hasFile('qr_code')) {
            $this->deleteQrCode($gateway->qr_code);
            $params['qr_code'] = $this->uploadQrCode($request->file('qr_code'));
        }
    }

    protected function uploadQrCode($file): string
    {
        return Storage::putFile('public/gateway/qr_codes', $file);
    }

    protected function deleteQrCode(?string $path): void
    {
        if ($path && Storage::exists($path)) {
            Storage::delete($path);
        }
    }
}